--[[
Copyright (C) GtX (Andy), 2021

Author: GtX | Andy
Date: 12.04.2021
Revision: FS25-02

Contact:
https://forum.giants-software.com
https://github.com/GtX-Andy

Important:
Not to be added to any mods / maps or modified from its current release form.
No changes are to be made to this script without permission from GtX | Andy

Darf nicht zu Mods / Maps hinzugefügt oder von der aktuellen Release-Form geändert werden.
An diesem Skript dürfen ohne Genehmigung von GtX | Andy keine Änderungen vorgenommen werden
]]

SupplyTransportManager = {}

local SupplyTransportManager_mt = Class(SupplyTransportManager)
local consoleCommandsGtX = StartParams.getIsSet("consoleCommandsGtX") -- Extra GtX console commands

function SupplyTransportManager.new(buildId, versionString, modName, modDirectory)
    local self = setmetatable({}, SupplyTransportManager_mt)

    self.isServer = g_server ~= nil
    self.isClient = g_client ~= nil

    self.buildId = buildId
    self.versionString = versionString

    self.modName = modName
    self.modDirectory = modDirectory

    -- Testing and content creator commands for SP or Hosted MP only. Use -cheats or -consoleCommandsGtX start parameter
    if self.isServer and (consoleCommandsGtX or g_addCheatCommands) then
        addConsoleCommand("gtxSupplyTransportMissionsSetDebugState", "Toggle debug mode", "consoleCommandSetDebugState", self, "isEnabled")
        addConsoleCommand("gtxSupplyTransportMissionsSetMaxInstances", "Set the maximum number of mission instances", "consoleCommandSetMaxNumInstances", self, "maxNumInstances; refreshMissions")
        addConsoleCommand("gtxSupplyTransportMissionsRefreshMissions", "Generate missions for testing", "consoleCommandRefreshMissions", self, "keepExisting; numberToGenerate; debugGeneration")
    end

    return self
end

function SupplyTransportManager:loadMapData(xmlFile, key, baseDirectory)
    if not self.loadedMapData then
        -- Only in Multiplayer in case a 'sellingStation' is removed while a mission is active.
        if self.isServer and g_currentMission.missionDynamicInfo.isMultiplayer then
            g_messageCenter:subscribe(MessageType.UNLOADING_STATIONS_CHANGED, self.onUnloadingStationsChanged, self)
        end
    else
        Logging.devError("[SupplyTransportManager] Map data was already loaded or not deleted after last game load!")
        printCallstack()
    end

    self.loadedMapData = true

    return true
end

function SupplyTransportManager:delete()
    g_messageCenter:unsubscribeAll(self)

    if self.isServer and (consoleCommandsGtX or g_addCheatCommands) then
        removeConsoleCommand("gtxSupplyTransportMissionsSetDebugState")
        removeConsoleCommand("gtxSupplyTransportMissionsSetMaxInstances")
        removeConsoleCommand("gtxSupplyTransportMissionsRefreshMissions")
    end

    self.loadedMapData = false
end

function SupplyTransportManager:onUnloadingStationsChanged()
    if self.isServer then
        local unloadingStations = g_currentMission.storageSystem:getUnloadingStations()
        local name = SupplyTransportMission.NAME

        for _, mission in ipairs(g_missionManager.missions) do
            if mission:getMissionTypeName() == name then
                if mission.sellingStation and not unloadingStations[mission.sellingStation] then
                    if mission.onSellingStationRemoved ~= nil then
                        mission:onSellingStationRemoved(mission.sellingStation)
                    else
                        mission.sellingStationRemoved = true
                    end

                    if mission:getIsInProgress() then
                        Logging.info("[SupplyTransportMission] Selling station '%s' used for contract has been deleted, contract cancelled!", mission:getSellingStationName())
                    elseif mission:getIsReadyToStart() then
                        Logging.info("[SupplyTransportMission] Selling station '%s' used for contract has been deleted, contract removed from list!", mission:getSellingStationName())
                    end
                end
            end
        end
    end
end

function SupplyTransportManager:getMaxNumInstances()
    local data = g_missionManager:getMissionTypeDataByName(SupplyTransportMission.NAME)

    return (data ~= nil and data.maxNumInstances) or SupplyTransportMission.DEFAULT_MAX_NUM_INSTANCES
end

function SupplyTransportManager:getIsDefaultFillType(fillTypeIndex)
    local data = g_missionManager:getMissionTypeDataByName(SupplyTransportMission.NAME)

    if data ~= nil and data.defaultFillTypes ~= nil then
        return data.defaultFillTypes[fillTypeIndex] ~= nil
    end

    return false
end

function SupplyTransportManager:getIsInvalidFillType(fillTypeIndex)
    local data = g_missionManager:getMissionTypeDataByName(SupplyTransportMission.NAME)

    if data ~= nil and data.invalidFillTypes ~= nil then
        if data.mapInvalidFillTypes ~= nil then
            return data.invalidFillTypes[fillTypeIndex] ~= nil, data.mapInvalidFillTypes[fillTypeIndex] ~= nil
        end

        return data.invalidFillTypes[fillTypeIndex] ~= nil, false
    end

    return false, false
end

function SupplyTransportManager:consoleCommandSetDebugState(isEnabled)
    if g_server == nil then
        return "'gtxSupplyTransportMissionsSetDebugState' is a server only command"
    end

    if isEnabled ~= nil then
        isEnabled = Utils.stringToBoolean(isEnabled)
    else
        isEnabled = not SupplyTransportMission.DEBUG_ENABLED
    end

    SupplyTransportMission.DEBUG_ENABLED = isEnabled

    return "SupplyTransportMission - Debug = " .. tostring(isEnabled)
end

function SupplyTransportManager:consoleCommandSetMaxNumInstances(maxNumInstances, refreshMissions)
    if g_server == nil then
        return "'gtxSupplyTransportMissionsSetMaxInstances' is a server only command"
    end

    local data = g_missionManager:getMissionTypeDataByName(SupplyTransportMission.NAME)

    if data == nil then
        return "Failed to find valid mission data!"
    end

    local currentMaxInstances = data.maxNumInstances

    maxNumInstances = math.clamp(tonumber(maxNumInstances) or 1, 1, SupplyTransportMission.MAX_NUM_INSTANCES or 15)

    if currentMaxInstances ~= maxNumInstances then
        data.maxNumInstances = maxNumInstances

        if Utils.stringToBoolean(refreshMissions) then
            print(string.format("Max instances has been changed from %d to %d", currentMaxInstances, maxNumInstances))
            return self:consoleCommandRefreshMissions("false", maxNumInstances, "false")
        end

        return string.format("Max instances has been changed from %d to %d", currentMaxInstances, maxNumInstances)
    end

    return "No changes detected, current max instances " .. tostring(currentMaxInstances)
end

function SupplyTransportManager:consoleCommandRefreshMissions(keepExisting, numberToGenerate, debugGeneration)
    if g_server == nil then
        return "'gtxSupplyTransportMissionsRefreshMissions' is a server only command"
    end

    local numAdded = 0
    local numRemoved = 0
    local numAvailable = 0
    local activeMissions = 0

    local name = SupplyTransportMission.NAME
    local missionManager = g_missionManager

    keepExisting = Utils.stringToBoolean(keepExisting)

    for _, mission in ipairs(missionManager.missions) do
        if mission:getMissionTypeName() == name then
            if not mission:getWasRunning() then
                if not keepExisting then
                    missionManager:markMissionForDeletion(mission)
                    numRemoved += 1
                else
                    numAvailable += 1
                end
            else
                activeMissions += 1
            end
        end
    end

    missionManager:updateMissions() -- Update missions marked for delete or no longer valid

    local missionType = missionManager:getMissionType(name)
    local classObject = missionType.classObject
    local debugEnabled = SupplyTransportMission.DEBUG_ENABLED

    if Utils.stringToBoolean(debugGeneration) then
        SupplyTransportMission.DEBUG_ENABLED = true
    end

    numberToGenerate = tonumber(numberToGenerate or 10)

    while numAdded < numberToGenerate do
        if classObject.tryGenerateMission == nil then
            break
        end

        local mission = classObject.tryGenerateMission()

        if mission == nil then
            break
        end

        missionManager:registerMission(mission, missionType)
        numAdded += 1
    end

    SupplyTransportMission.DEBUG_ENABLED = debugEnabled

    return string.format("%d contracts have been cleared, %d new contracts created.", numRemoved, numAdded)
end
